<?php
// Este archivo maneja la actualización de la prioridad de una reparación
require_once '../config/database.php'; // ASEGÚRATE QUE LA RUTA SEA CORRECTA

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

header('Content-Type: application/json'); // Indicamos que la respuesta será JSON

// 1. Verificación de solicitud y datos
if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_POST['id_reparacion'], $_POST['nueva_prioridad'])) {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'message' => 'Solicitud inválida o datos faltantes.']);
    exit;
}

$id_reparacion = (int)$_POST['id_reparacion'];
$nueva_prioridad = $_POST['nueva_prioridad'];

// 2. Validación de la prioridad (por seguridad)
if (!in_array($nueva_prioridad, ['Baja', 'Media', 'Alta'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Prioridad no válida.']);
    exit;
}

try {
    // 3. Consulta para actualizar la prioridad
    $sql = "UPDATE reparaciones SET prioridad = ? WHERE id = ?";
    $stmt = $conexion->prepare($sql);
    
    if ($stmt === false) {
        throw new Exception('Error al preparar la consulta: ' . $conexion->error);
    }
    
    // El primer parámetro 's' es string (para la prioridad), 'i' es integer (para el id)
    $stmt->bind_param("si", $nueva_prioridad, $id_reparacion);
    
    if ($stmt->execute()) {
        if ($stmt->affected_rows > 0) {
    // 4. Éxito en la actualización: Incluimos la nueva prioridad para el mensaje de éxito en JS
    echo json_encode(['success' => true, 'message' => 'Prioridad actualizada con éxito.', 'nueva_prioridad' => $nueva_prioridad]);
} else {
            // No se encontraron cambios o la reparación no existe
            echo json_encode(['success' => false, 'message' => 'No se encontraron cambios o la reparación no existe.']);
        }
    } else {
        throw new Exception('Error al ejecutar la actualización: ' . $stmt->error);
    }

    $stmt->close();
    
} catch (Exception $e) {
    // 5. Manejo de errores internos
    http_response_code(500); // Internal Server Error
    echo json_encode(['success' => false, 'message' => 'Error interno del servidor: ' . $e->getMessage()]);
}

$conexion->close();
?>